/***************************************************************************/
/*                                                                         */
/*  irdr.c                                                                 */
/*                                                                         */
/*  Copyright (C) 2012 Monotype Imaging Inc. All rights reserved.          */ 
/*                                                                         */
/***************************************************************************/

#include "irdr.h"
#include "fs_function.h"
#include FT_INTERNAL_TRUETYPE_TYPES_H

#ifdef FT_UNUSED
#undef FT_UNUSED
#endif
#define FT_UNUSED(arg) (void)arg /* to suppress Lint errors */

#define ITC_INTERNAL_MAGIC 0x69726472 /* 'irdr' */

/*
 * defining this macro will display iType and Freetype outline information
 */
#undef DUMP_OUTLINES

#ifdef DUMP_OUTLINES
void itype_rdr_dump_itype_outline( FS_OUTLINE *outl, char *msg );
void itype_rdr_dump_freetype_outline( FT_Outline *outl, char *msg );
#endif

#ifdef TXTC_BUILDS
#define LOG_TAG "itype"
#include <utils/Log.h>
#define ANDROID_LOG_INFO  4 /* write to logcat */
#endif

const FT_Raster_Funcs itype_raster =
{
    FT_GLYPH_FORMAT_OUTLINE,
    (FT_Raster_New_Func)         itype_raster_new,
    (FT_Raster_Reset_Func)       itype_raster_reset,
    (FT_Raster_Set_Mode_Func)    itype_raster_set_mode,
    (FT_Raster_Render_Func)      itype_raster_render,
    (FT_Raster_Done_Func)        itype_raster_done
};

/*************************************************************************/
/*                                                                       */
/*                      itype_rdr_module_init()                          */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( FT_Error )
itype_rdr_module_init( FT_Renderer  render )
{
    FT_Error error = ITYPE_RDR_Err_Ok;
    FT_UNUSED( render );
    return error;
}

/*************************************************************************/
/*                                                                       */
/*                        itype_rdr_module_done()                        */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( void )
itype_rdr_module_done( FT_Renderer render )
{
    FT_UNUSED( render );
}

/*************************************************************************/
/*                                                                       */
/*                  itype_rdr_calc_num_of_oncurve_points()               */
/*                                                                       */
/*************************************************************************/
FT_Int
itype_rdr_calc_num_of_oncurve_points( FT_Outline *outl )
{
    short n_points;
    FT_Int points, i, j;

    if( !outl )
        return 0;

    n_points = outl->n_points;

    for( i=0, j=0, points=0; i < n_points; ++i, j++ )
    {
        if( (outl->tags[j] & 3) == FT_CURVE_TAG_ON )
            points++;
    }
    return points;
}

/*************************************************************************/
/*                                                                       */
/*                      itype_rdr_font_name_new()                        */
/*                                                                       */
/*************************************************************************/
FILECHAR*
itype_rdr_font_name_new( FT_Face ttface )
{
    FT_Error error = ITYPE_RDR_Err_Ok;
    FT_Memory memory;
    FILECHAR *font_name = NULL;
    size_t len = 0;

    if( ttface == NULL )
        return NULL;

    memory = FT_FACE_MEMORY( ttface );

    /*
     * concatenate the family_name and the style_name together
     *  to create the font name
     */
    if( ttface->family_name != NULL )
    {
        len = ft_strlen( ttface->family_name ) + 2;
        if( ttface->style_name != NULL )
            len += ft_strlen( ttface->style_name );
    }

    if( len > 0 )
    {
        if( !FT_ALLOC( font_name, len + 1 ))
        {
            strcat( font_name, ttface->family_name );
            if( ttface->style_name != NULL )
            {
                strcat( font_name, " " );
                strcat( font_name, ttface->style_name );
            }
        }
    }
    return font_name;
}

/*************************************************************************/
/*                                                                       */
/*                     itype_rdr_font_name_free()                        */
/*                                                                       */
/*************************************************************************/
FS_VOID
itype_rdr_font_name_free( FT_Face ttface, FILECHAR *font_name )
{
    TT_Face face = (TT_Face)ttface;
    FT_Memory memory = FT_FACE_MEMORY( face );

    if( font_name )
        FT_MEM_FREE( font_name );
}

/*************************************************************************/
/*                                                                       */
/*                   itype_rdr_validate_itype_size()                     */
/*                                                                       */
/*************************************************************************/
void
itype_rdr_validate_itype_size( FT_Size size )
{
    if( size )
    {
        if( (size->metrics.x_ppem > ITYPE_SCALE_UPPER_LIMIT) ||
            (size->metrics.y_ppem > ITYPE_SCALE_UPPER_LIMIT) )
        {
            size->metrics.x_ppem = ITYPE_SCALE_UPPER_LIMIT;
            size->metrics.y_ppem = ITYPE_SCALE_UPPER_LIMIT;
            size->metrics.x_scale = ( ITYPE_SCALE_UPPER_LIMIT * size->face->units_per_EM ) << 6;
            size->metrics.y_scale = ( ITYPE_SCALE_UPPER_LIMIT * size->face->units_per_EM ) << 6;
        }
        else if( (size->metrics.x_ppem < ITYPE_SCALE_LOWER_LIMIT) ||
                 (size->metrics.y_ppem < ITYPE_SCALE_LOWER_LIMIT) )
        {
            size->metrics.x_ppem = ITYPE_SCALE_LOWER_LIMIT;
            size->metrics.y_ppem = ITYPE_SCALE_LOWER_LIMIT;
            size->metrics.x_scale = ( ITYPE_SCALE_LOWER_LIMIT * size->face->units_per_EM ) << 6;
            size->metrics.y_scale = ( ITYPE_SCALE_LOWER_LIMIT * size->face->units_per_EM ) << 6;
        }
    }
}

/*************************************************************************/
/*                                                                       */
/*                      itype_rdr_done_itype_outline()                   */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( void )
itype_rdr_done_itype_outline( FS_STATE *pState, FS_OUTLINE *outl )
{
    if( outl )
    {
        if( outl->y )
            FS_free( pState, outl->y );

        if( outl->x )
            FS_free( pState, outl->x );

        if( outl->type )
            FS_free( pState, outl->type );

        FS_free( pState, outl );
    }
}

/*************************************************************************/
/*                                                                       */
/*                   itype_rdr_set_itype_outline_bbox()                  */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( void )
itype_rdr_set_itype_outline_bbox(FS_OUTLINE *outl)
{
    FS_FIXED lo_x,hi_x,lo_y,hi_y,x,y;
    FS_SHORT i,n_points = outl->np;

    if( n_points == 0 )
        outl->lo_x = outl->hi_x = outl->lo_y = outl->hi_y = 0;
    else
    {
        FS_FIXED *px, *py;
        px = outl->x;
        lo_x = hi_x = px[0];
        py = outl->y;
        lo_y = hi_y = py[0];
        for( i=1; i<n_points; i++ )
        {
            x = px[i];
            if( x > hi_x ) hi_x = x;
            if( x < lo_x ) lo_x = x;
            y = py[i];
            if( y > hi_y ) hi_y = y;
            if( y < lo_y ) lo_y = y;
        }
        outl->lo_x = lo_x;
        outl->lo_y = lo_y;
        outl->hi_x = hi_x;
        outl->hi_y = hi_y;
    }
}

/*************************************************************************/
/*                                                                       */
/*                       itype_rdr_make_itype_outline()                  */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( FS_OUTLINE* )
itype_rdr_make_itype_outline( FS_STATE *FS_state_ptr, FT_Outline *ft_outl )
{
    FS_SHORT     i, j, k, l, pre_conic, pre_cubic;
    FS_SHORT     quad_started;
    FS_OUTLINE  *it_outl;
    FS_ULONG     n_points, n_types, n_off_curv_pts;
    FS_FIXED    *x,*y, pre_x, pre_y, cur_x, cur_y;
    FS_FIXED     x_start, y_start;

    /* for freetype generated FT_Outline, the insert point between two conic types is missing */
    /* we need count them in.                                                                 */
    n_off_curv_pts = 0;
    pre_conic = 0;
    for( i=0; i<ft_outl->n_points; i++ )
    {
        if( FT_CURVE_TAG(ft_outl->tags[i]) == FT_CURVE_TAG_CONIC )
        {
            if( pre_conic )
                n_off_curv_pts += 1;
            
            pre_conic = 1;
        }
        else
            pre_conic = 0;
    }

    n_points = ft_outl->n_points + ft_outl->n_contours + n_off_curv_pts;
    n_types  = itype_rdr_calc_num_of_oncurve_points(ft_outl) + ft_outl->n_contours + + n_off_curv_pts;

#ifdef DUMP_OUTLINES
    itype_rdr_dump_freetype_outline( ft_outl, "This is the input freetype outline." );
#endif
    /* make a iType outline from the local ft_outline */
    it_outl = (FS_OUTLINE *)FS_calloc( FS_state_ptr, sizeof( FS_OUTLINE ));
    if( it_outl == NULL )
        return NULL;

    it_outl->type = (FS_BYTE *)FS_malloc( FS_state_ptr, sizeof( FS_BYTE ) * n_types );
    if( it_outl->type == NULL )
    {
        itype_rdr_done_itype_outline( FS_state_ptr, it_outl );
        return NULL;
    }

    it_outl->x = (FS_FIXED *)FS_malloc( FS_state_ptr, sizeof( FS_FIXED ) * n_points );
    if( it_outl->x == NULL )
    {
        itype_rdr_done_itype_outline( FS_state_ptr, it_outl );
        return NULL;
    }

    it_outl->y = (FS_FIXED *)FS_malloc( FS_state_ptr, sizeof( FS_FIXED ) * n_points );
    if( it_outl->y == NULL )
    {
        itype_rdr_done_itype_outline( FS_state_ptr, it_outl );
        return NULL;
    }

    it_outl->size = (FS_LONG)sizeof(FS_OUTLINE);
    it_outl->nc   = (FS_SHORT) ft_outl->n_contours;
    it_outl->num  = (FS_SHORT) n_types;
    it_outl->np   = (FS_SHORT) n_points;
    it_outl->polarity = 1;

    x = it_outl->x;
    y = it_outl->y;
    x_start = (FS_FIXED) ( ft_outl->points[0].x << 10 );
    y_start = (FS_FIXED) ( ft_outl->points[0].y << 10 );
    quad_started = 0;

    /* ft_outl generated by freetype driv may have a CONIC type at 1st pont that */
    /* will cause problem in it_outline conversion, we need move it to the end.  */

    /* do 1st contour here */
    if( FT_CURVE_TAG(ft_outl->tags[0]) == FT_CURVE_TAG_CONIC )
    {
        FT_Pos temp_x    = ft_outl->points[0].x;
        FT_Pos temp_y    = ft_outl->points[0].y;
        char   temp_type = ft_outl->tags[0] & 3;

        for( j=0; j < (ft_outl->contours[0]); j++ ) 
        {
            ft_outl->points[j].x = ft_outl->points[j+1].x;
            ft_outl->points[j].y = ft_outl->points[j+1].y;
            ft_outl->tags[j]     = ft_outl->tags[j+1] & 3;
        }
        ft_outl->points[j].x = temp_x; 
        ft_outl->points[j].y = temp_y; 
        ft_outl->tags[j]     = temp_type;
    }

    /* do 2nd and following contours */
    for( i=1; i < ft_outl->n_contours; i++ )
    {
        if( FT_CURVE_TAG(ft_outl->tags[ft_outl->contours[i-1]+1]) == FT_CURVE_TAG_CONIC )
        {
            FT_Pos temp_x    = ft_outl->points[ft_outl->contours[i-1]+1].x;
            FT_Pos temp_y    = ft_outl->points[ft_outl->contours[i-1]+1].y;
            char   temp_type = ft_outl->tags[ft_outl->contours[i-1]+1] & 3;

            for( j=ft_outl->contours[i-1]+1; j<ft_outl->contours[i]; j++ ) 
            {
                ft_outl->points[j].x = ft_outl->points[j+1].x;
                ft_outl->points[j].y = ft_outl->points[j+1].y;
                ft_outl->tags[j]     = ft_outl->tags[j+1] & 3;
            }
            ft_outl->points[j].x = temp_x; 
            ft_outl->points[j].y = temp_y; 
            ft_outl->tags[j]     = temp_type;
        }
    }

#ifdef DUMP_OUTLINES
    itype_rdr_dump_freetype_outline( ft_outl, "After the 1st point CONIC fix:" );
#endif
    /* since not a space character, convert the data (and add end points) */
    /* i = FreeType point index, k = FreeType contour index */
    /* j = iType type index, l = iType point index          */
    pre_conic = 0;
    pre_cubic = 0;
    pre_x = 0;
    pre_y = 0;

    for( i = 0, j = 0, k = 0, l = 0; i < ft_outl->n_points; i++ )
    {
        char outline_tag = ft_outl->tags[i];

        switch( FT_CURVE_TAG(outline_tag) )
        {
        case FT_CURVE_TAG_ON:
            pre_conic = 0;
            if( i == 0 )
            {
                *x++ = x_start = (FS_FIXED)(ft_outl->points[i].x << 10);
                *y++ = y_start = (FS_FIXED)(ft_outl->points[i].y << 10);
                it_outl->type[j++] = (FS_BYTE) FS_MOVETO;
                l++;
            }
            else if( i == ft_outl->contours[k] + 1 )
            {
                *x++ = x_start; /* close the contour */
                *y++ = y_start;
                l++;

                if( pre_cubic )
                {
                    it_outl->type[j++] = (FS_BYTE)FS_CUBETO;
                    pre_cubic = 0;
                }
                else
                {
                    if( quad_started == 0 )
                        it_outl->type[j++] = (FS_BYTE) FS_LINETO;
                    else
                        quad_started = 0;
                }

                /* now the moveto point */
                *x++ = x_start = (FS_FIXED) ( ft_outl->points[i].x << 10 );
                *y++ = y_start = (FS_FIXED) ( ft_outl->points[i].y << 10 );
                it_outl->type[j++] = (FS_BYTE) FS_MOVETO;
                k++;
                l++;
            }
            else
            {
                if( pre_cubic == 2 )
                {
                    it_outl->type[j++] = (FS_BYTE)FS_CUBETO;
                    pre_cubic = 0;
                }
                else
                {
                    if( !quad_started )
                       it_outl->type[j++] = (FS_BYTE) FS_LINETO;
                }

                *x++ = (FS_FIXED) ( ft_outl->points[i].x << 10 );
                *y++ = (FS_FIXED) ( ft_outl->points[i].y << 10 );
                l++;
                quad_started = 0;
            }
            break;

        case FT_CURVE_TAG_CONIC:
            if( pre_conic ) /* if we see two conic together, insert a CURVE_ON point */
            {
                FS_FIXED tmp_x, tmp_y; 

                tmp_x = (FS_FIXED)( ft_outl->points[i].x << 10 );
                tmp_y = (FS_FIXED)( ft_outl->points[i].y << 10 );
                cur_x = (tmp_x + pre_x) >> 1; /* insert CURVE_ON point_x */
                cur_y = (tmp_y + pre_y) >> 1; /* insert CURVE_ON point_y */
               
                *x++ = cur_x; /* this is the inserting ON_CURVE point */
                *y++ = cur_y;
                *x++ = tmp_x; /* this is the conic point */
                *y++ = tmp_y;

                it_outl->type[j++] = (FS_BYTE) FS_QUADTO;
                quad_started = 1;
            }
            else
            {
                pre_conic = 1;
                pre_x = cur_x = (FS_FIXED)( ft_outl->points[i].x << 10 );
                pre_y = cur_y = (FS_FIXED)( ft_outl->points[i].y << 10 );
                it_outl->type[j++] = (FS_BYTE) FS_QUADTO;
                quad_started = 1;
                *x++ = cur_x;
                *y++ = cur_y;
            }        
            l++;
            break;

        case FT_CURVE_TAG_CUBIC:
            pre_conic = 0;
            pre_cubic++;
            *x++ = (FS_FIXED)(ft_outl->points[i].x << 10);
            *y++ = (FS_FIXED)(ft_outl->points[i].y << 10);
            l++;
            break;
        default:
            break;
        }
    }

    /* close the last contour */
    *x = x_start;
    *y = y_start;
    if( pre_cubic )
        it_outl->type[j++] = (FS_BYTE)FS_CUBETO;
    else
    {
        if( !quad_started )
            it_outl->type[j++] = (FS_BYTE) FS_LINETO;
    }

    /* number of actual iType outline types and points */
    it_outl->num = j;
    it_outl->np = l + 1;

    if (!(ft_outl->flags &  FT_OUTLINE_IGNORE_DROPOUTS))
        it_outl->outl_flag |= OUTL_FLAGS_SCANCTRL;

    itype_rdr_set_itype_outline_bbox(it_outl);

    it_outl->dx = (it_outl->hi_x - it_outl->lo_x);
    it_outl->dy = (it_outl->hi_y - it_outl->lo_y);
    it_outl->i_dx = FS_ROUND(it_outl->dx);
    it_outl->i_dy = FS_ROUND(it_outl->dy);

#ifdef DUMP_OUTLINES
    itype_rdr_dump_itype_outline( it_outl, "converted itype outline." );
#endif
    return it_outl;
}

/*************************************************************************/
/*                                                                       */
/*                         itype_render_generic()                        */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( FT_Error )
itype_render_generic( FT_Renderer      render,
                      FT_GlyphSlot     slot,
                      FT_Render_Mode   mode,
                      const FT_Vector *origin )
{
    FT_Error             error;
    FT_Raster_Params     params;
    FT_Memory            memory;
    FT_Raster_User_Params_ITC user_params;
    iTypeRaster_Instance *raster_inst;

    FT_UNUSED(origin);
    FT_UNUSED(mode);

    if( slot == NULL )
        return ITYPE_RDR_Err_Cannot_Render_Glyph;

    raster_inst = (iTypeRaster_Instance *)(void *)render->raster;
    if ( raster_inst == NULL)
        return ITYPE_RDR_Err_Cannot_Render_Glyph;

    memory = FT_MODULE_MEMORY( &render->root );

    if( slot->internal->flags & FT_GLYPH_OWN_BITMAP )
    {
        FT_FREE( slot->bitmap.buffer );
        slot->internal->flags &= ~FT_GLYPH_OWN_BITMAP;
    }

    /* get the flags from the MTI extension */
    if( (slot->face != NULL) && VALID_MTI_EXTENSION(slot->face->extensions) )
    {
        MTI_Info ext = (MTI_Info)slot->face->extensions;

        if ( ext != NULL )
            raster_inst->load_flags = ext->load_flags;
        else
        {
            ITC_PRINTF_DEBUG(("Failed to get MTI extension in itype_render_generic()\n"));
        }
    }

    /* set up the user params */
    FT_MEM_SET(&user_params, 0, sizeof(FT_Raster_User_Params_ITC));
    user_params.library = slot->library;
    user_params.magic = ITC_INTERNAL_MAGIC;

    /* this will be overridden by the value in the params structure appended to the end of the outline points */
    user_params.renderWithEdge = 0/*FALSE*/;

    params.source = &slot->outline;
    params.target = &slot->bitmap;
    params.user = &user_params;

    if( mode == FT_RENDER_MODE_MONO )
        params.flags = FT_RASTER_FLAG_DEFAULT;
    else
        params.flags = FT_RASTER_FLAG_AA;      /* regular */

    error = itype_raster_render( (iTypeRaster_Instance *)(void *)render->raster, &params );
    if( error != ITYPE_RDR_Err_Ok )
        return error;

    slot->internal->flags |= FT_GLYPH_OWN_BITMAP;
    slot->format = FT_GLYPH_FORMAT_BITMAP;
    slot->bitmap_left = user_params.bitmap_left;
    slot->bitmap_top  = user_params.bitmap_top;

    /* set freetype metrics, all data in FT_Glyph_Metrics are in 26.6 format. */
    if( !(raster_inst->load_flags & FT_LOAD_NO_SCALE) )
    {
        slot->metrics.width = slot->bitmap.width << 6;
        slot->metrics.height= slot->bitmap.rows  << 6;
    }

    return error;
}

/*************************************************************************/
/*                                                                       */
/*                          itype_rdr_render()                           */
/*                                                                       */
/*************************************************************************/
static FT_Error
itype_rdr_render( FT_Renderer render,
                  FT_GlyphSlot slot,
                  FT_Render_Mode mode,
                  const FT_Vector *origin )
{
    return itype_render_generic( render, slot, mode, origin );
}

/*************************************************************************/
/*                                                                       */
/*                          itype_rdr_transform ()                       */
/*                                                                       */
/*************************************************************************/
static FT_Error
itype_rdr_transform( FT_Renderer       render,
                     FT_GlyphSlot      slot,
                     const FT_Matrix*  matrix,
                     const FT_Vector*  delta )
{
    FT_Error error = ITYPE_RDR_Err_Ok;

    if( slot->format != render->glyph_format )
    {
        error = ITYPE_RDR_Err_Invalid_Argument;
        return error;
    }

    if( matrix )
        FT_Outline_Transform( &slot->outline, matrix );

    if( delta )
        FT_Outline_Translate( &slot->outline, delta->x, delta->y );

    return error;
}

/*************************************************************************/
/*                                                                       */
/*                          itype_rdr_get_cbox()                         */
/*                                                                       */
/*************************************************************************/
static void
itype_rdr_get_cbox( FT_Renderer   render,
                    FT_GlyphSlot  slot,
                    FT_BBox*      cbox )
{
    FT_UNUSED(render);
    FT_Outline_Get_CBox( &slot->outline, cbox );
}

/*************************************************************************/
/*                                                                       */
/*                           itype_rdr_set_mode()                        */
/*                                                                       */
/*************************************************************************/
static FT_Error
itype_rdr_set_mode( FT_Renderer  render,
                    FT_ULong     mode_tag,
                    FT_Pointer   data )
{
    FT_Error  error = ITYPE_RDR_Err_Ok;
    FT_UNUSED(data);
    FT_UNUSED(mode_tag);
    FT_UNUSED(render);
    return error;
}

#ifdef FS_EDGE_RENDER
/*************************************************************************/
/*                                                                       */
/*                  itype_rdr_set_ADFRenderAttrs()                       */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( FS_VOID )
itype_rdr_set_ADFRenderAttrs( ADFRenderAttrs  *adfRenderAttrs,
                                     FS_USHORT ppem )
{
    adfRenderAttrs->penX = 0 /*0.0*/;
    adfRenderAttrs->penY = 0 /*0.0*/;
    adfRenderAttrs->pointSize = ppem << 16; /* 16.16 value */
    adfRenderAttrs->dpi = 72;
    adfRenderAttrs->scaleX = 65536 /*1.0*/;
    adfRenderAttrs->scaleY = 65536 /*1.0*/;
    adfRenderAttrs->rotationPtX = 0;
    adfRenderAttrs->rotationPtY = 0;
    adfRenderAttrs->rotationAngle = 0;
    adfRenderAttrs->displayMode = ADF_REND_MODE_CRT;
    adfRenderAttrs->gridFitType = 0;
    adfRenderAttrs->outsideCutoff = 0;
    adfRenderAttrs->insideCutoff = 0;
    adfRenderAttrs->gamma = 65536;
    adfRenderAttrs->useColorReduction = 0;
    adfRenderAttrs->colorReductionAmt = 0;
}
#endif /* FS_EDGE_RENDER */

/*************************************************************************/
/*                                                                       */
/*                            itype_raster_new()                         */
/*                                                                       */
/*************************************************************************/
static int
itype_raster_new( void *memory, iTypeRaster_Instance **araster )
{
    FT_Error error = ITYPE_RDR_Err_Ok;
    iTypeRaster_Instance *the_raster;

    if( !FT_NEW( the_raster ) )
    {
        the_raster->memory = memory;
        the_raster->load_flags = FT_LOAD_DEFAULT;
        the_raster->unused1 = 1;
        the_raster->unused2 = 2;

        *araster = the_raster;
    }

    return error;
}

/*************************************************************************/
/*                                                                       */
/*                          itype_raster_reset()                         */
/*                                                                       */
/*************************************************************************/
static void
itype_raster_reset( iTypeRaster_Instance raster, unsigned char* pool_base, unsigned char pool_size )
{
    FT_UNUSED( raster );
    FT_UNUSED( pool_base );
    FT_UNUSED( pool_size );
}

/*************************************************************************/
/*                                                                       */
/*                          itype_raster_set_mode ()                     */
/*                                                                       */
/*************************************************************************/
static int
itype_raster_set_mode( iTypeRaster_Instance *raster, unsigned long mode, void* args )
{
    FT_UNUSED( raster );
    FT_UNUSED( mode );
    FT_UNUSED( args );
    return 0;
}

/*************************************************************************/
/*                                                                       */
/*                         itype_raster_done()                           */
/*                                                                       */
/*************************************************************************/
static void
itype_raster_done( iTypeRaster_Instance  *raster )
{
    FT_Memory memory = (FT_Memory) raster->memory;
    FT_FREE( raster );
}

FT_LOCAL_DEF( FS_STATE * )
get_itype_state(FT_Library   library)
{
    FS_STATE *retVal = NULL;

    FT_Module itype_font_driver = FT_Get_Module( library, ITYPE_DRIVER_NAME );
    if ( itype_font_driver != NULL )
    {
        Generic_Module_Data *GMD = itype_font_driver->generic.data;
        if ( GMD != NULL )
            retVal = GMD->FS_state_ptr;
    }

    return retVal;
}

typedef struct itype_params_
{
    FS_FIXED     bold_pct;
    FS_FIXED     stroke_pct;
    FS_USHORT    cur_font;
    FS_ULONG     flags;
    FS_USHORT    lpm;
    LFNT         *cur_lfnt;
    SFNT         *cur_sfnt;
    TYPESET      cur_typeset;
#ifdef FS_EDGE_RENDER
    ADF_U32      adfGridFitType;
#endif

} itype_params;

FT_LOCAL_DEF (void)
cache_itype_params(itype_params *cache, FS_STATE *state)
{
    /* save the current settings */
    cache->bold_pct = state->bold_pct;
    cache->stroke_pct = state->stroke_pct;
    cache->cur_font = state->cur_font;
    cache->flags = state->flags;
    cache->lpm = state->lpm;
    cache->cur_lfnt = state->cur_lfnt;
    cache->cur_sfnt = state->cur_sfnt;
    cache->cur_typeset = state->cur_typeset;
#ifdef FS_EDGE_RENDER
    cache->adfGridFitType = state->adfGridFitType;
#endif
}

FT_LOCAL_DEF (void)
restore_itype_params(itype_params *cache, FS_STATE *state)
{
    state->bold_pct = cache->bold_pct;
    state->stroke_pct = cache->stroke_pct;
    state->cur_font = cache->cur_font;
    state->flags = cache->flags;
    state->lpm = cache->lpm;
    state->cur_lfnt = cache->cur_lfnt;
    state->cur_sfnt = cache->cur_sfnt;
    state->cur_typeset = cache->cur_typeset;
#ifdef FS_EDGE_RENDER
    state->adfGridFitType = cache->adfGridFitType;
#endif
}

FT_LOCAL_DEF (FT_Error) setup_itype_params(FT_Library library, FS_STATE *state )
{
    Generic_Module_Data *GMD;
    FT_Module itype_font_driver = FT_Get_Module( library, ITYPE_DRIVER_NAME );

    if (itype_font_driver == NULL)
        return ITYPE_RDR_Err_Cannot_Render_Glyph;

    GMD = itype_font_driver->generic.data;
    if (GMD == NULL)
        return ITYPE_RDR_Err_Cannot_Render_Glyph;

    state->bold_pct = 0;
    state->stroke_pct = 0;
    state->cur_font = 0;
    state->flags = 0;

    SYS_MEMSET(&GMD->ss.lfnt,0,sizeof(LFNT));
    SYS_MEMSET(&GMD->ss.sfnt,0,sizeof(SFNT));
    SYS_MEMSET(&GMD->ss.tfnt,0,sizeof(TFNT));
    SYS_MEMSET(&GMD->ss.cfnt,0,sizeof(CFNT));

    state->cur_lfnt = &GMD->ss.lfnt;
    state->cur_sfnt = &GMD->ss.sfnt;
#ifdef FS_EDGE_RENDER
    state->adfGridFitType = ADF_GRID_FIT_NONE;
    itype_rdr_set_ADFRenderAttrs(&GMD->ss.tfnt.adfRenderAttrs, state->lpm);
#endif
    GMD->ss.typeset.tfntarray = &GMD->ss.tfnt;
    GMD->ss.typeset.tfntarray->cfnt = &GMD->ss.cfnt;
    state->cur_typeset = GMD->ss.typeset;

    return ITYPE_RDR_Err_Cannot_Render_Glyph;
}

FT_LOCAL_DEF( int ) create_bitmap( iTypeRaster_Instance *raster, FS_GLYPHMAP *itype_glyph, FT_Bitmap *ft_bitmap )
{
    FT_Error error = ITYPE_RDR_Err_Ok;
    FT_Memory memory = raster->memory; /* memory = FT_MODULE_MEMORY( &render->root ); */
    int buffer_size;

    ft_bitmap->rows  = itype_glyph->height;
    ft_bitmap->width = itype_glyph->width;
    ft_bitmap->num_grays = (short)(1<<itype_glyph->bitsPerPixel);
    ft_bitmap->palette_mode  = 0; /* unused */
    ft_bitmap->palette       = 0; /* unused */

    switch ( itype_glyph->bitsPerPixel )
    {
        case 0: /* FALLTHROUGH */
        case 1:
            ft_bitmap->pixel_mode = FT_PIXEL_MODE_MONO;
            ft_bitmap->pitch = itype_glyph->bpl;
            buffer_size = itype_glyph->height*itype_glyph->bpl;
            break;
        case 2:
            ft_bitmap->pixel_mode = FT_PIXEL_MODE_GRAY2;
            ft_bitmap->pitch = itype_glyph->width;
            buffer_size = itype_glyph->height*(itype_glyph->width);
            break;
        case 4:
            ft_bitmap->pixel_mode = FT_PIXEL_MODE_GRAY4;
            ft_bitmap->pitch = itype_glyph->width;
            buffer_size = itype_glyph->height * (itype_glyph->width );
            break;
        case 8:
            ft_bitmap->pixel_mode = FT_PIXEL_MODE_GRAY;
            ft_bitmap->pitch = itype_glyph->bpl;
            buffer_size = itype_glyph->height * itype_glyph->bpl;
            break;
        default:
            ft_bitmap->pixel_mode = FT_PIXEL_MODE_NONE;
            ft_bitmap->pitch = itype_glyph->bpl;
            buffer_size = itype_glyph->height*itype_glyph->bpl;
            break;
    }

    if( FT_ALLOC( ft_bitmap->buffer, buffer_size ) )
    {
        return ITYPE_RDR_Err_Cannot_Render_Glyph;
    }

    if ( buffer_size == itype_glyph->height*itype_glyph->bpl )
    {
        ft_memcpy(ft_bitmap->buffer, itype_glyph->bits, buffer_size);
    }
    else if ( itype_glyph->bitsPerPixel == 2 )
    {
        int i, j;
        FS_BYTE *src;
        unsigned char *dest;
        FS_BYTE bit2Tobit8[4] = {0, 0x55, 0xAA, 0xFF};

        src = itype_glyph->bits;
        dest = ft_bitmap->buffer;

        for ( i = 0; i < ft_bitmap->rows; i++)
        {
            for ( j = 0; j < ft_bitmap->width; j++ )
            {
                FS_BYTE val = src[j/4];

                switch (j%4)
                {
                default:
                case 0:
                    val = (val & 0xC0) >> 6;
                    break;
                case 1:
                    val = (val & 0x30) >> 4;
                    break;
                case 2:
                    val = (val & 0x0C) >> 2;
                    break;
                case 3:
                    val = (val & 0x03);
                    break;
                }

                dest[j] = bit2Tobit8[val];
            }
            src += itype_glyph->bpl;
            dest += ft_bitmap->pitch;
        }
    }
    else if ( itype_glyph->bitsPerPixel == 4 )
    {
        int i, j;
        FS_BYTE *src;
        unsigned char *dest;
        FS_BYTE bit4Tobit8[16] = {0, 17, 34, 51, 68, 85, 102, 119, 136, 153, 170, 187, 204, 221, 238, 255};

        src = itype_glyph->bits;
        dest = ft_bitmap->buffer;

        for ( i = 0; i < ft_bitmap->rows; i++)
        {
            for ( j = 0; j < ft_bitmap->width; j++ )
            {
                FS_BYTE val;
                val = (src[j/2] & (j%2 ? 0x0F : 0xF0)) >> (j%2 ? 0 : 4);
                dest[j] = bit4Tobit8[val];
            }
            src += itype_glyph->bpl;
            dest += ft_bitmap->pitch;
        }
    }

    return ITYPE_RDR_Err_Ok;
}

FT_LOCAL_DEF( void )
copy_glyph_to_target_buf( FS_GLYPHMAP *itype_glyph, FT_Outline *ft_outl, FT_Bitmap *ft_bitmap )
{
    unsigned char *pft, *pit;
    int i, j;
    int num_rows, num_cols, num_bytes, dest_left_offset, src_left_offset;
    FT_UInt cbox_width, cbox_height;
    FT_BBox cbox;
    int origPixelMode = ft_bitmap->pixel_mode;

    /* set the pixel mode */
    switch ( itype_glyph->bitsPerPixel )
    {
        case 0: /* FALLTHROUGH */
        case 1:
            ft_bitmap->pixel_mode = FT_PIXEL_MODE_MONO;
            break;
        case 2:
            ft_bitmap->pixel_mode = FT_PIXEL_MODE_GRAY2;
            break;
        case 4:
            ft_bitmap->pixel_mode = FT_PIXEL_MODE_GRAY4;
            break;
        case 8:
            ft_bitmap->pixel_mode = FT_PIXEL_MODE_GRAY;
            break;
        default:
            ft_bitmap->pixel_mode = FT_PIXEL_MODE_NONE;
            break;
    }

    /* set pointers */
    pft = ft_bitmap->buffer;
    pit = itype_glyph->bits;

    /* init */
    num_bytes = dest_left_offset = src_left_offset = 0;

    /* determine if the itype glyph fits in the given buffer and if it needs to be shifted */

#define ITC_RAST_DEBUG(x) /*ITC_PRINTF_DEBUG(x)*/

    ITC_RAST_DEBUG(("iType glyph   -> width:%d height:%d lo_x:%d, hi_y:%d\n", itype_glyph->width, itype_glyph->height, itype_glyph->lo_x, itype_glyph->hi_y));
    ITC_RAST_DEBUG(("FT bitmap     -> width:%d height:%d\n", ft_bitmap->width, ft_bitmap->rows));

    /* Get the cbox -- NOTE:: it is assumed that the outline has been
       appropriately shifted already */
    FT_Outline_Get_CBox(ft_outl, &cbox);

    ITC_RAST_DEBUG(("Outline CBox  ->  top:%1.2f bottom:%1.2f left:%1.2f right:%1.2f\n", cbox.yMax / 64.0, cbox.yMin / 64.0, cbox.xMin / 64.0, cbox.xMax / 64.0));

    cbox.xMin = FT_PIX_FLOOR( cbox.xMin );
    cbox.yMin = FT_PIX_FLOOR( cbox.yMin );
    cbox.xMax = FT_PIX_CEIL( cbox.xMax );
    cbox.yMax = FT_PIX_CEIL( cbox.yMax );

    ITC_RAST_DEBUG(("dilated CBox  ->  top:%1.2f bottom:%1.2f left:%1.2f right:%1.2f\n", cbox.yMax / 64.0, cbox.yMin / 64.0, cbox.xMin / 64.0, cbox.xMax / 64.0));

    cbox_width  = (FT_UInt)( ( cbox.xMax - cbox.xMin ) >> 6 );
    cbox_height = (FT_UInt)( ( cbox.yMax - cbox.yMin ) >> 6 );

    ITC_RAST_DEBUG(("CBox dimens.  -> width:%d height:%d\n", cbox_width, cbox_height));

    if ((cbox_width != (FT_UInt)ft_bitmap->width) || (cbox_height != (FT_UInt)ft_bitmap->rows))
    {
        ITC_RAST_DEBUG(("The sizes do not match!\n"));
    }

    /* adjust placement in the y direction */
    num_rows = ft_bitmap->rows;

    if( itype_glyph->height > ft_bitmap->rows )
    {
        /* the itype glyph is taller than the available space, so
           we move the starting pointer in the itype glyph down, which will clip off the top of the glyph */
        pit += (itype_glyph->height - ft_bitmap->rows)*itype_glyph->bpl;
        ITC_RAST_DEBUG(("The glyph will be clipped in the vertical direction!!!\n"));
    }
    else if( itype_glyph->height < ft_bitmap->rows )
    {
        FS_LONG fTop = (cbox.yMax >> 6);

        /* the itype glyph is shorter than the available space 
           Figure out where the top of the glyph should be in the output buffer */

        num_rows = itype_glyph->height;

        if( fTop - itype_glyph->hi_y > 1 )
        {
            int maxRowsToAdvance = ft_bitmap->rows - num_rows;
            int rowsToAdvance = fTop - itype_glyph->hi_y - 1;

            if (rowsToAdvance > maxRowsToAdvance)
                rowsToAdvance = maxRowsToAdvance;
            pft += (rowsToAdvance)*ft_bitmap->pitch;
        }
    }

    /* adjust placement in the x direction */
    num_cols = ft_bitmap->width;

    if(itype_glyph->bpl < ft_bitmap->pitch)
        num_bytes = itype_glyph->bpl;
    else
        num_bytes = ft_bitmap->pitch;

    if ( num_cols < itype_glyph->width )
    {
        ITC_RAST_DEBUG(("The glyph will be clipped on the right side!\n"));
    }
    else if ( num_cols > itype_glyph->width )
    {
        /* need to figure out where horizontally the itype glyph should be copied to the output buffer */

        dest_left_offset = itype_glyph->lo_x;
        src_left_offset = 0;
        num_cols = itype_glyph->width;

        if (dest_left_offset < 0)
            dest_left_offset = 0;

        if ( dest_left_offset + itype_glyph->width > ft_bitmap->width)
            dest_left_offset = ft_bitmap->width - itype_glyph->width;
    }

    ITC_RAST_DEBUG(("\n"));

    /* copy the bits */
    if ( itype_glyph->bitsPerPixel == 8 )
    {
        if ( (origPixelMode == FT_PIXEL_MODE_MONO) || (origPixelMode == FT_PIXEL_MODE_NONE) )
        {
            /* convert down from 8bpp to 1bpp */
            for( i=0; i < num_rows; i++ )
            {
                int k = 0;
                unsigned char target = 0;

                for( j=0; (j<num_cols) && (k < num_bytes); j++ )
                {
                    if (pit[j] > 0x7f)
                        target |= (1 << (8 - ((j%8)+1)));

                    if (((j+1)%8 == 0) || (j == (num_cols-1)))
                    {
                        pft[k] |= target;
                        k++;
                        target = 0;
                    }
                }
                pft += ft_bitmap->pitch;
                pit += itype_glyph->bpl;
            }
        }
        else
        {
            for( i=0; i<num_rows; i++ )
            {
                for( j=0; j<num_cols; j++ )
                    pft[j+dest_left_offset] |= pit[j+src_left_offset];

                pft += ft_bitmap->pitch;
                pit += itype_glyph->bpl;
            }
        }
    }
    else if ( (itype_glyph->bitsPerPixel == 1) || (itype_glyph->bitsPerPixel == 0) )
    {
        if ( origPixelMode == FT_PIXEL_MODE_GRAY )
        {
            /* up-convert from monochrome to 8 bit */
            for( i=0; i<num_rows; i++ )
            {
                unsigned char *byte_ptr = pit;
                int bit_index = 7;

                for( j=0; j<num_cols; j++ )
                {
                    /* The outline's image is OR-ed to the target bitmap */
                    pft[j+dest_left_offset] |= ((((*byte_ptr) & (1 << bit_index)) == 0) ? 0 : 255);

                    bit_index--;
                    if (bit_index < 0)
                    {
                        bit_index = 7;
                        byte_ptr++;
                    }
                }
                pft += ft_bitmap->pitch;
                pit += itype_glyph->bpl;
            }
        }
        else
        {
            for( i=0; i<num_rows; i++ )
            {
                for( j=0; j<num_bytes; j++ )
                    pft[j] |= pit[j];

                pft += ft_bitmap->pitch;
                pit += itype_glyph->bpl;
            }
        }
    }
    else if ( itype_glyph->bitsPerPixel == 2 )
    {
        FS_BYTE *src;
        unsigned char *dest;
        src = pit;
        dest = pft;

        if ( origPixelMode == FT_PIXEL_MODE_GRAY )
        {
            FS_BYTE bit2Tobit8[4] = {0, 0x55, 0xAA, 0xFF};

            for ( i = 0; i < num_rows; i++)
            {
                for ( j = 0; j < num_cols; j++ )
                {
                    FS_BYTE val = src[j/4];

                    switch (j%4)
                    {
                    default:
                    case 0:
                        val = (val & 0xC0) >> 6;
                        break;
                    case 1:
                        val = (val & 0x30) >> 4;
                        break;
                    case 2:
                        val = (val & 0x0C) >> 2;
                        break;
                    case 3:
                        val = (val & 0x03);
                        break;
                    }

                    dest[j+dest_left_offset] |= bit2Tobit8[val];
                }
                src += itype_glyph->bpl;
                dest += ft_bitmap->pitch;
            }
        }
        else
        {
            /* down convert from 2bpp to 1bpp */
            for( i=0; i<num_rows; i++ )
            {
                int k = 0;
                unsigned char target = 0;

                for( j=0; (j < num_cols) && (k < num_bytes); j++ )
                {
                    FS_BYTE val = src[j/4];

                    switch (j%4)
                    {
                    default:
                    case 0:
                        val = (val & 0xC0) >> 6;
                        break;
                    case 1:
                        val = (val & 0x30) >> 4;
                        break;
                    case 2:
                        val = (val & 0x0C) >> 2;
                        break;
                    case 3:
                        val = (val & 0x03);
                        break;
                    }

                    if (val > 1)
                        target |= (1 << (8 - ((j%8)+1)));

                    if (((j+1)%8 == 0) || (j == (num_cols-1)))
                    {
                        dest[k] |= target;
                        k++;
                        target = 0;
                    }
                }

                dest += ft_bitmap->pitch;
                src += itype_glyph->bpl;
            }
        }
    }
    else if ( itype_glyph->bitsPerPixel == 4 )
    {
        FS_BYTE *src;
        unsigned char *dest;
        FS_BYTE bit4Tobit8[16] = {0, 17, 34, 51, 68, 85, 102, 119, 136, 153, 170, 187, 204, 221, 238, 255};

        src = pit;
        dest = pft;

        if ( origPixelMode == FT_PIXEL_MODE_GRAY )
        {
            for ( i = 0; i < num_rows; i++)
            {
                for ( j = 0; j < num_cols; j++ )
                {
                    FS_BYTE val;
                    val = (src[j/2] & (j%2 ? 0x0F : 0xF0)) >> (j%2 ? 0 : 4);
                    dest[j+dest_left_offset] |= bit4Tobit8[val];
                }
                src += itype_glyph->bpl;
                dest += ft_bitmap->pitch;
            }
        }
        else
        {
            /* down convert from 4bpp to 1bpp */
            for( i=0; i<num_rows; i++ )
            {
                int k = 0;
                unsigned char target = 0;

                for( j=0; (j < num_cols) && (k < num_bytes); j++ )
                {
                    FS_BYTE val = (src[j/2] & (j%2 ? 0x0F : 0xF0)) >> (j%2 ? 0 : 4);

                    if (val > 7)
                        target |= (1 << (8 - ((j%8)+1)));

                    if (((j+1)%8 == 0) || (j == (num_cols-1)))
                    {
                        dest[k] |= target;
                        k++;
                        target = 0;
                    }
                }

                dest += ft_bitmap->pitch;
                src += itype_glyph->bpl;
            }
        }
    }
}

/*************************************************************************/
/*                                                                       */
/*                          itype_raster_render()                        */
/*                                                                       */
/*************************************************************************/

static int
itype_raster_render( iTypeRaster_Instance *raster, const FT_Raster_Params *params )
{
    FS_USHORT    glyph_type;
    FT_Raster_User_Params_ITC *user_params;
    FS_GLYPHMAP *itype_glyph;
    FS_STATE    *FS_state_ptr;
    FT_Outline  *ft_outl;
    FT_Bitmap   *ft_bitmap;
    FS_OUTLINE  *it_outl;
    itype_params saveParams;
    int retval = ITYPE_RDR_Err_Ok;

    if( (raster == NULL) || (params == NULL) || (params->user == NULL) )
        return ITYPE_RDR_Err_Cannot_Render_Glyph;

    user_params = (FT_Raster_User_Params_ITC *)params->user;

    /* validate the params */
    if ( (user_params->magic != ITC_CLIENT_MAGIC) && (user_params->magic != ITC_INTERNAL_MAGIC) )
        return ITYPE_RDR_Err_Cannot_Render_Glyph;


    ft_outl     = (FT_Outline *)params->source;
    ft_bitmap   = (FT_Bitmap *)params->target;

    if( ft_outl->n_points == 0 )   /* space character, nothing to render */
        return ITYPE_RDR_Err_Ok;

    /* If the outline was not generated by iType then don't allow iType to render it. */
    if( !(ft_outl->flags & FT_OUTLINE_FROM_ITYPE_DRV) )
        return ITYPE_RDR_Err_Cannot_Render_Glyph;

    /* Get the iType STATE from the GMD */
    FS_state_ptr = get_itype_state(user_params->library);
    if( FS_state_ptr == NULL )
        return ITYPE_RDR_Err_Cannot_Render_Glyph;

    /* Construct an iType outline structure from the Freetype Outline */
    it_outl = itype_rdr_make_itype_outline( FS_state_ptr, ft_outl );
    if( it_outl == NULL )
        return ITYPE_RDR_Err_Cannot_Render_Glyph;

    /*
     * Calling FS_set_outline allows the caller to specify an externally generated
     * outline so that iType will use this outline during rendering. Once the
     * outline is set it remains in effect until a new outline or NULL has been specified.
     * When an external outline has been set any FS_get_glyphmap(), FS_get_bitmap(),
     * FS_get_graymap() or FS_get_phased() call will return a glyph rendered using
     * the specified outline regardless of the glyph ID specified in the call.
     */
    if( SUCCESS != FS_set_outline( FS_state_ptr, it_outl, FALSE, 0 ) )
    {
        itype_rdr_done_itype_outline( FS_state_ptr, it_outl );
        return ITYPE_RDR_Err_iType_Set_Outline_Failed;
    }

    /* save the current iType state settings */
    cache_itype_params( &saveParams, FS_state_ptr );

    /* set the temporary settings into the state */
    FS_state_ptr->lpm = (FS_USHORT)((FT_Raster_User_Params_ITC *)(void*)&ft_outl->points[ft_outl->n_points])->ppem; /* ppem */
    setup_itype_params( user_params->library, FS_state_ptr );

    if( it_outl->outl_flag &  OUTL_FLAGS_SCANCTRL )
        FS_state_ptr->flags |= FLAGS_FIX_DROPOUTS;

    /* initialize the glyph type */
#if (ITYPE_RENDER_BITDEPTH==8)
    glyph_type = FS_MAP_GRAYMAP8;
#elif (ITYPE_RENDER_BITDEPTH==4)
    glyph_type = FS_MAP_GRAYMAP4;
#elif (ITYPE_RENDER_BITDEPTH==2)
    glyph_type = FS_MAP_GRAYMAP2;
#else
#error invalid bit depth
#endif

    if( (params->flags & FT_RASTER_FLAG_AA) != 0 )
    {
        /* gray glyph */
#ifdef FS_EDGE_RENDER
        FT_Raster_User_Params_ITC *paramsToApply;
        FT_Bool edgeRender = FALSE;

        /* user settings will take precedence over those in the outline */
        if (user_params->magic == ITC_CLIENT_MAGIC)
        {
            paramsToApply = user_params;
            if (paramsToApply->renderWithEdge == 1/*TRUE*/)
                edgeRender = TRUE;
        }
        else
        {
            paramsToApply = (FT_Raster_User_Params_ITC *)(void *)&ft_outl->points[ft_outl->n_points];
            if (paramsToApply->renderWithEdge == 1/*TRUE*/)
                edgeRender = TRUE;
        }

        if ( edgeRender == TRUE )
        {
#if (ITYPE_RENDER_BITDEPTH==8)
            glyph_type = FS_MAP_EDGE_GRAYMAP8;
#elif (ITYPE_RENDER_BITDEPTH==4)
            glyph_type = FS_MAP_EDGE_GRAYMAP4;
#elif (ITYPE_RENDER_BITDEPTH==2)
            glyph_type = FS_MAP_EDGE_GRAYMAP2;
#else
#error invalid bit depth
#endif

            FS_set_csm( FS_state_ptr, paramsToApply->insideCutoff,
                                      paramsToApply->outsideCutoff,
                                      paramsToApply->gammaValue );

            FS_set_csm_adjustments( FS_state_ptr, paramsToApply->sharpness_offset,
                                                  paramsToApply->sharpness_slope,
                                                  paramsToApply->thickness_offset,
                                                  paramsToApply->thickness_slope );

            FS_set_flags( FS_state_ptr, FLAGS_DEFAULT_CSM_ON );
        }

#ifdef TXTC_BUILDS
        if (((FT_Raster_User_Params_ITC *)&ft_outl->points[ft_outl->n_points])->renderWithEdge == 2) /* test font */
           __android_log_print(ANDROID_LOG_INFO, "ITYPE_EDGERENDERING", "%ld", 1L);
#endif

#endif
    }
    else
    {
        /* monochrome glyph */
        glyph_type = FS_MAP_BITMAP;

        if( !(FS_state_ptr->flags & FLAGS_FIX_DROPOUTS) )
            FS_state_ptr->flags |= FLAGS_FIX_DROPOUTS;
    }

    /* get the glyph */
    itype_glyph = FS_get_glyphmap( FS_state_ptr, 0, glyph_type );
    if( itype_glyph == NULL )
    {
        restore_itype_params( &saveParams, FS_state_ptr );
        itype_rdr_done_itype_outline( FS_state_ptr, it_outl );
        FS_set_flags( FS_state_ptr, FLAGS_NO_EFFECT ); /* this clears out the outline previously set */
        FS_set_outline( FS_state_ptr, NULL, FALSE, 0 );
        return ITYPE_RDR_Err_iType_Error;
    }

    /* return glyph origin in case they need it */
    user_params->bitmap_top  = itype_glyph->hi_y + 1;
    user_params->bitmap_left = itype_glyph->lo_x;

    /* restore the iType state settings */
    restore_itype_params( &saveParams, FS_state_ptr );

    /* if this is an empty bitmap, we can allocate and set the buffer directly */
    if( (ft_bitmap->rows == 0) && (ft_bitmap->pitch == 0) && (ft_bitmap->buffer == NULL) )
    {
        retval = create_bitmap( raster, itype_glyph, ft_bitmap );
    }
    else
    {
        /* The client has preallocated the buffer, 'OR' the new glyph buffer into existing input bitmap buffer */
        copy_glyph_to_target_buf( itype_glyph, ft_outl, ft_bitmap );
    }

    FS_free_char( FS_state_ptr, itype_glyph );
    itype_rdr_done_itype_outline( FS_state_ptr, it_outl );
    FS_set_flags( FS_state_ptr, FLAGS_NO_EFFECT ); /* this clears out the outline previously set */
    FS_set_outline( FS_state_ptr, NULL, FALSE, 0 );

    return retval;
}

/*************************************************************************/
/*                                                                       */
/*                          itype_renderer_class                         */
/*                                                                       */
/*************************************************************************/
FT_CALLBACK_TABLE_DEF
const FT_Renderer_Class  itype_renderer_class =
{
    {
      FT_MODULE_RENDERER,                               /* bitflags describing the module */
      sizeof( FT_RendererRec ),                         /* the size of one module in bytes */

      ITYPE_RENDERER,                                   /* renderer module name */
      0x10000L,                                         /* renderer module version == 1.0 */
      0x20000L,                                         /* renderer module requires FreeType 2.0 or above */

      0,                                                /* renderer module specific interface */

      (FT_Module_Constructor)itype_rdr_module_init,     /* function to initialize not create the renderer */
      (FT_Module_Destructor) itype_rdr_module_done,     /* function to finalize not destroy the renderer */
      (FT_Module_Requester)  0                          /* queries the renderer for a specific interface by name */
    },

    FT_GLYPH_FORMAT_OUTLINE,                            /* the glyph image format this renderer handles */

    (FT_Renderer_RenderFunc)   itype_rdr_render,        /* function used to render the glyph slot into a bitmap */
    (FT_Renderer_TransformFunc)itype_rdr_transform,     /* function used to transform the bitmap */
    (FT_Renderer_GetCBoxFunc)  itype_rdr_get_cbox,      /* function used to get the bounding box */
    (FT_Renderer_SetModeFunc)  itype_rdr_set_mode,      /* function used to pass additional parameters */

    (FT_Raster_Funcs*) &itype_raster                    /* function used to render the glyph slot into a bitmap */
};


/*************************************************************************/
/*                                                                       */
/*                     itype_rdr_dump_itype_outline()                    */
/*                                                                       */
/*************************************************************************/
#ifdef DUMP_OUTLINES
void
itype_rdr_dump_itype_outline( FS_OUTLINE *outl, char *msg )
{
    int i, j;
    FS_FIXED *x,*y;

    if(!outl)
        return;

    printf( "\n" );
    printf("----------------------------------------------------------\n");
    printf( "iType Renderer dump FS_OUTLINE: %s\n", msg );
    printf("----------------------------------------------------------\n");

    /* header */
    printf( "num_types = %d, num_points = %d, num_contours = %d\n", outl->num, outl->np, outl->nc );
    printf("xmin=%5.5f  xmax=%5.5f  ymin=%5.5f  ymax=%5.5f\n",
            outl->lo_x/65536.0, outl->hi_x/65536.0, 
            outl->lo_y/65536.0, outl->hi_y/65536.0 );

    printf( "i_dx=%d  i_dy=%d  dx=%5.5f  dy=%5.5f\n",
             outl->i_dx, outl->i_dy, outl->dx/65536.0, outl->dy/65536.0 );
    printf("----------------------------------------------------------\n");

    /* now the curve data */
    x = outl->x;
    y = outl->y;

    for( i=0, j=0; i<outl->num; i++ )
    {
        switch( outl->type[i] & 0x7f)
        {
            case FS_MOVETO:
                printf("%d %d %12.5f %12.5f moveto\n", j++, i, *x++ / 65536.0, *y++ / 65536.0);
                break;
            case FS_LINETO:
                printf("%d %d %12.5f %12.5f lineto\n", j++, i,*x++ / 65536.0, *y++ / 65536.0);
                break;
            case FS_QUADTO:
                printf("%d %d %12.5f %12.5f\n", j++, i,*x++ / 65536.0, *y++ / 65536.0);
                printf("%d %d %12.5f %12.5f quadto\n", j++, i,*x++ / 65536.0, *y++ / 65536.0);
                break;
            case FS_CUBETO:
                printf("%d %d %12.5f %12.5f\n", j++, i,*x++ / 65536.0, *y++ / 65536.0);
                printf("%d %d %12.5f %12.5f\n", j++, i,*x++ / 65536.0, *y++ / 65536.0);
                printf("%d %d %12.5f %12.5f cubeto\n", j++, i,*x++ / 65536.0, *y++ / 65536.0);
                break;
        }
    }
    printf( "\n" );
    fflush( stdout );
}
#endif

/*************************************************************************/
/*                                                                       */
/*                     itype_rdr_dump_freetype_outline()                 */
/*                                                                       */
/*************************************************************************/
#ifdef DUMP_OUTLINES
void
itype_rdr_dump_freetype_outline( FT_Outline *outl, char *msg )
{
    int i = 0;

    printf( "\n" );
    printf( "----------------------------------------------------------\n" );
    printf( "iType Renderer dump FT_Outline: %s\n", msg );
    printf( "----------------------------------------------------------\n" );
    printf( "num_points = %d, num_contours = %d\n", outl->n_points, outl->n_contours );
    printf( "index          x              y         type\n" );
    printf( "----------------------------------------------------------\n" );

    for( i = 0; i < outl->n_points; i++ )
    {
        printf( "  %d     %12.5f     %12.5f", i, outl->points[i].x/64.0, outl->points[i].y/64.0 );

        switch(FT_CURVE_TAG(outl->tags[i]))
        {
        case FT_CURVE_TAG_ON:
            printf("  %s\n","FT_CURVE_TAG_ON");
            break;
        case FT_CURVE_TAG_CONIC:
            printf("  %s\n","FT_CURVE_TAG_CONIC");
            break;
        case FT_CURVE_TAG_CUBIC:
            printf("  %s\n","FT_CURVE_TAG_CUBIC");
            break;
        default:
            printf("  %s\n","Unknown");
            break;
        }
    }

    for( i = 0; i < outl->n_contours; i++ )
    {
        printf("\ncontour = %d  index = %d  x =%5.5f    y =%5.5f",i ,
                                   outl->contours[i],
                                   outl->points[outl->contours[i]].x/64.0,
                                   outl->points[outl->contours[i]].y/64.0);
    }
    printf("\n");
    fflush(stdout);
}
#endif

/* end of file */
